/*
 * main.cpp - control routine for Automaton
 *
 * This file is part of Automaton.
 *
 * Copyright (C) 2002, 2003
 * Paul Gettings, Dep't of Geology & Geophysics
 * University of Utah
 *
 * This file is released under the terms of the software
 * license in the file "LICENSE" in the root directory of
 * this package.  If this file is missing or corrupt, please
 * contact the author to receive a new copy.
 *
 * Automaton is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  Use at your
 * own risk; your mileage may vary.
 *
 * Suggestions, improvements, and bug reports welcome at
 * <gettings@mines.utah.edu>
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "autopick.h"
#include "segy.h"

void usage(void) {
  fprintf(stderr, "usage: autopick [opts] cmd_file pick_file output_file segy_file [segy_file ...]\n");
  fprintf(stderr, "where opts are:\n");
  fprintf(stderr, "  -s		swap bytes; this is a toggle, and defaults to ON\n");
  fprintf(stderr, "  -r		skip reel header in SEGY files\n");
  fprintf(stderr, "  -a		input data is in ASCII format, NOT SEGY!\n");
  fprintf(stderr, "  -d dbgfile	name for autopick debug file\n");
  fprintf(stderr, "  -h		print this help\n");
}

/* Main routine to read data and known pick, and then autopick
 * the rest of the data
 *
 * Reads data from SEGY file, and known picks for each shot from
 * known pick file.
 *
 * Processes each shot in isolation
 *
 * No parallelization!
 */
int main(int argc, char *argv[])
{
  autopickParameters parms;
  seismicData *S; // readSEGYMultiShot will allocate this
  knownPick known;
  char debugFile[2048];

  const char *optstring="srahd";

  FILE *out, *fp, *dbg;

  if(argc < 5) {
    usage();
    exit(1);
    }

  bool swapBytes=true, reelHeader=false, asciiFiles=false; 
  int c;
  strcpy(debugFile, "debug.out");
  while((c = getopt(argc, argv, optstring)) != -1) {
    switch(c) {
      case 's': swapBytes = !swapBytes; break;
      case 'r': reelHeader = true; break;
      case 'a': asciiFiles = true; break;
      case 'd': strncpy(debugFile, optarg, 2047);
                break;
      case 'h':
      case '?':
      default : usage(); exit(1); break;
      }
    }


  /* read algorithm parameters into autopickParameters struct */
  parms = grokCommandFile(argv[optind]);
  // check parameters
  if(parms.Threshhold <= 0 || parms.Threshhold > 1) { // Threshhold must be between (0,1]
    fprintf(stderr, "Threshhold value(%lf) silly. Abort.\n", parms.Threshhold); exit(1); }
  if(parms.AmpConst <= 0) { // Amplitude constant must be positive!
    fprintf(stderr, "AmpConst value(%lf) silly. Abort.\n", parms.AmpConst); exit(1); }
  if(parms.TimeConst <= 0) { // Time constant must be positive!
    fprintf(stderr, "TimeConst value(%lf) silly. Abort.\n", parms.TimeConst); exit(1); }
  if(parms.Velocity <= 0 || parms.Velocity > 100000) { // velocity in (0,100] km/s
    fprintf(stderr, "Velocity value(%lf) silly. Abort.\n", parms.Velocity); exit(1); }
  if(parms.MinVelocity <= 0 || parms.MinVelocity > 100000) { // velocity in (0,100] km/s
    fprintf(stderr, "MinVelocity value(%lf) silly. Abort.\n", parms.MinVelocity); exit(1); }
  if(parms.MaxVelocity <= 0 || parms.MaxVelocity > 100000) { // velocity in (0,100] km/s
    fprintf(stderr, "MaxVelocity value(%lf) silly. Abort.\n", parms.MaxVelocity); exit(1); }
  if(parms.Attenuation <= 0) { // attenuation between (0,Inf]
    fprintf(stderr, "Attenuation value(%lf) silly. Abort.\n", parms.Attenuation); exit(1); }
  if(parms.WaveTime <= 0 || parms.WaveTime > 100000) { // wave length between (0,100] sec
    fprintf(stderr, "WaveTime value(%lf) silly. Abort.\n", parms.WaveTime); exit(1); }

  /* open the output file */
  if(!strcmp(argv[optind+2], "-")) {
    out = stdout;
    }
  else if((out = fopen(argv[optind+2], "wt")) == NULL) {
    fprintf(stderr, "cannot open %s for pick output. Abort.\n", argv[4]);
    exit(1);
    }

  if((dbg = fopen(debugFile, "wt")) == NULL) {
    fprintf(stderr, "cannot open debug file '%s' for write. Abort.\n", debugFile);
    return(0);
    }

  /* read the seismic data */
  long i, ntrace, nshot;
  int idx;
  for(idx = optind+3; idx < argc; idx++) { // process each input file in turn
    if(asciiFiles) { // read data from our own ASCII format; only one shot per file
      if((S = (seismicData *)malloc(sizeof(seismicData))) == NULL) {
        fprintf(stderr, "cannot allocate memory for seismic data structure.\n");
        exit(1);
        }
      S[0] = readSeismicData(argv[idx]); // fn dies to system if error
      nshot = 1;
      }
    else { // SEGY files, maybe with many shots
      if((fp = fopen(argv[idx], "rb")) == NULL) {
	fprintf(stderr, "cannot open '%s' for data read.\n", argv[idx]);
	exit(1);
	}
      i = checkSEGY(fp, swapBytes, reelHeader, &ntrace, &nshot);
      S = readSEGYMultiShot(fp, nshot, swapBytes, reelHeader); // works for just one shot also
      fclose(fp);
      }

    fprintf(stderr, "#Processing %d shots for file '%s'.\n", nshot, argv[idx]);

    // do each shot in isolation; parallelize here?
    for(i=0; i<nshot; i++) {
      /* read the known pick - must be done after loading seismic data */
      known = readKnownPick(argv[optind+1], S[i], parms); // fn dies to system if error
      /* pick the arrivals */
      if(!autopick(S[i], parms, known, out, dbg)) { // good picks written to file
	exit(1);
	}
      killSeismic(&S[i]); // toss data for this shot
      }
    free(S); // free memory for next file
    }

  /* cleanup and exit */
  fclose(out); fclose(dbg);
  exit(0);
}

